﻿using System;
using System.IO;
using System.Security.Cryptography;

namespace Crypto
{
    class Program
    {
        // Listing 11-51. Zastosowanie strumienia szyfrującego.
        static void Main(string[] args)
        {
            byte[] key;
            byte[] iv;
            // Pobieramy odpowiedni klucz i inicjujemy wektor dla algorytmu.
            SelectKeyAndIV(out key, out iv);
            string superSecret = "Supertajny tekst.";
            Console.WriteLine(superSecret);
            string encryptedText = EncryptString(superSecret, key, iv);
            Console.WriteLine(encryptedText);
            string decryptedText = DecryptString(encryptedText, key, iv);
            Console.WriteLine(decryptedText);
            Console.ReadKey();
        }

        // Listing 11-52. Tworzenie klucza i wektora początkowego (IV).
        private static void SelectKeyAndIV(out byte[] key, out byte[] iv)
        {
            var algorithm = TripleDES.Create();
            algorithm.GenerateIV();
            algorithm.GenerateKey();
            key = algorithm.Key;
            iv = algorithm.IV;
        }

        // Listing 11-53. Szyfrowanie łańcucha znaków.
        private static string EncryptString(string plainText, byte[] key, byte[] iv)
        {
            // Tworzymy dostawcę usług kryptograficznych dla algorytmu TripleDES.
            var serviceProvider = new TripleDESCryptoServiceProvider();

            using (MemoryStream memoryStream = new MemoryStream())
            using (var cryptoStream = new CryptoStream(
                                            memoryStream,
                                            serviceProvider.CreateEncryptor(key, iv),
                                            CryptoStreamMode.Write))
            using (StreamWriter writer = new StreamWriter(cryptoStream))
            {
                // Zapisujemy tekst w strumieniu kryptograficznym, co powoduje
                // jego zaszyfrowanie.
                writer.Write(plainText);
                // Upewniamy się, że dane zostały opróżnione do strumienia.
                writer.Flush();
                // Musimy także zadbać o to, by sam strumień został opróżniony, co zapisze
                // jego zawartość w strumieniu docelowym, gdyż w przeciwnym razie możemy
                // utracić część danych...
                cryptoStream.FlushFinalBlock();

                // A teraz chcemy pobrać to, co strumień kryptograficzny zapisał w naszym 
                // strumieniu MemoryStream.
                return GetCipherText(memoryStream);
            }
        }

        // Listing 11-54. Konwersja danych na Base64
        private static string GetCipherText(MemoryStream memoryStream)
        {
            byte[] buffer = memoryStream.ToArray();
            return System.Convert.ToBase64String(buffer, 0, buffer.Length);
        }

        // Listing 11-55. Deszyfracja
        private static string DecryptString(string cipherText, byte[] key, byte[] iv)
        {
            // Tworzymy dostawcę usług kryptograficznych dla algorytmu TripleDES.
            var serviceProvider = new TripleDESCryptoServiceProvider();

            // Dekodujemy zaszyfrowane dane z Base64 do postaci binarnej.
            byte[] cipherTextBytes = Convert.FromBase64String(cipherText);

            // Tworzymy strumień MemoryStream operujący na tych danych.
            using (MemoryStream memoryStream = new MemoryStream(cipherTextBytes))
            // Tworzymy strumień kryptograficzny operujący na strumieniu MemoryStream,
            // używając przy tym konkretnego algorytmu (ze ściśle określonym
            // kluczem i wektorem początkowym).
            using (var cryptoStream =
                new CryptoStream(
                    memoryStream,
                    serviceProvider.CreateDecryptor(key, iv),
                    CryptoStreamMode.Read))
            // W końcu tworzymy strumień do odczytu operujący na strumieniu
            // kryptograficznym i używamy go do pobrania oryginalnego,
            // odszyfrowanego tekstu.
            using (StreamReader reader = new StreamReader(cryptoStream))
            {
                return reader.ReadToEnd();
            }
        }
    }
}
